<!--
title: Serverless Framework - Variables - S3 Objects
description: How to reference AWS S3 Objects
short_title: Serverless Variables - S3 Objects
keywords: ['Serverless Framework', 'S3', 'AWS', 'Variables']
-->

<!-- DOCS-SITE-LINK:START automatically generated  -->

### [Read this on the main serverless docs site](https://www.serverless.com/framework/docs/guides/variables/aws/s3)

<!-- DOCS-SITE-LINK:END -->

# Configuration options

| Option                 | Required |  Type  |              Default               | Description                      |
| ---------------------- | :------: | :----: | :--------------------------------: | :------------------------------- |
| `region`               |    No    | String | Inherited from parent AWS resolver | AWS region                       |
| `bucketName`           |    No    | String |                                    | AWS S3 bucket name               |
| `objectKey`            |    No    | String |                                    | AWS S3 object key                |
| `serverSideEncryption` |    No    | String |                                    | Server-side encryption algorithm |

## Examples

### Default Configuration

In this example, the `awsAccount1` provider is used to fetch an object from an S3 bucket using the default region associated with your deployment.
This setup is useful when the S3 bucket is located in the same region as your deployment, and you simply want to retrieve an S3 object without any additional configuration.

```yaml
stages:
  default:
    resolvers:
      awsAccount1:
        type: aws

functions:
  hello:
    handler: handler.hello
    description: ${awsAccount1:s3:myBucket/myKey}
```

### S3 URI

Here, the S3 object is referenced using the full S3 URI format (`s3://myBucket/myKey`).
This format is helpful when you’re accustomed to working with S3 URIs or when other parts of your setup involve S3 URIs directly.
The resolver automatically identifies the bucket and key based on the URI.

```yaml
stages:
  default:
    resolvers:
      awsAccount1:
        type: aws

functions:
  hello:
    handler: handler.hello
    description: ${awsAccount1:s3:s3://myBucket/myKey}
```

### S3 ARN

This example shows how to reference an S3 object using its full ARN (Amazon Resource Name).
The ARN format is useful when your deployment involves resources that rely on ARNs for identification.
The resolver automatically identifies the bucket and key based on the ARN.

```yaml
stages:
  default:
    resolvers:
      awsAccount1:
        type: aws

functions:
  hello:
    handler: handler.hello
    description: ${awsAccount1:s3:arn:aws:s3:::myBucket/myKey}
```

### Bucket Name provided in the resolver configuration

In this case, the `myBucket` resolver is configured with both the bucket name and a custom region (`eu-west-1`).
The S3 object key is specified dynamically (`myKey`) in the service configuration.
This setup is ideal when the bucket is located in a different region from your deployment,
and you need flexibility in specifying different object keys without redefining the bucket name and region each time.

```yaml
stages:
  default:
    resolvers:
      awsAccount1:
        type: aws
        region: us-west-2
        myBucket:
          type: s3
          region: eu-west-1
          bucketName: myBucket

functions:
  hello:
    handler: handler.hello
    description: ${awsAccount1:myBucket:myKey}
```

### Bucket Name and Object Key provided in the resolver configuration

This example fully defines the bucket name and object key in the resolver configuration itself.
The function configuration simply references `${awsAccount1:myFile}`,
which fetches the `myKey` object from `myBucket` in the `eu-west-1` region.
This setup is beneficial when both the bucket and object key are consistent and do not require customization across different parts of your service configuration.
By centralizing this information, you avoid repetitive definitions and keep your configuration cleaner and more maintainable.

```yaml
stages:
  default:
    resolvers:
      awsAccount1:
        type: aws
        region: us-west-2
        myFile:
          type: s3
          region: eu-west-1
          bucketName: myBucket
          objectKey: myKey

functions:
  hello:
    handler: handler.hello
    description: ${awsAccount1:myFile}
```

### Server-Side Encryption

By default, S3 interactions inherit the [bucket’s default encryption settings](https://docs.aws.amazon.com/AmazonS3/latest/userguide/bucket-encryption.html). However, there are scenarios where you
may need to explicitly specify a server-side encryption algorithm to comply with security policies. For example,
a bucket policy may require all objects to be encrypted using AES256.

In such cases, you can use the `serverSideEncryption` configuration option. This ensures that
the `x-amz-server-side-encryption` header is included in all S3 interactions performed through the resolver,
avoiding access issues caused by bucket policies.

Consider the following bucket policy, which denies `PutObject` operations unless a valid encryption algorithm is specified:

```json
{
  "Version": "2012-10-17",
  "Id": "Policy",
  "Statement": [
    {
      "Sid": "EnforceEncryptionMethod",
      "Effect": "Deny",
      "Principal": "*",
      "Action": "s3:PutObject",
      "Resource": [
        "arn:aws:s3:::<BUCKET_NAME>",
        "arn:aws:s3:::<BUCKET_NAME>/*"
      ],
      "Condition": {
        "StringNotEquals": {
          "s3:x-amz-server-side-encryption": ["AES256"]
        }
      }
    }
  ]
}
```

If the `serverSideEncryption` option is not configured, accessing this bucket will fail with an error like this:

```
AccessDenied: User: <ARN> is not authorized to perform: s3:PutObject on resource: "arn:aws:s3:::OBJECT_KEY" with an explicit deny in a resource-based policy
```

To resolve this, include the `serverSideEncryption` option in your configuration to specify the required encryption algorithm.
For example:

```yaml
stages:
  default:
    resolvers:
      awsAccount1:
        type: aws
        myEncryptedBucket:
          type: s3
          bucketName: myBucket
          serverSideEncryption: AES256

functions:
  hello:
    handler: handler.hello
    description: ${awsAccount1:myEncryptedBucket:myKey}
```

# Classic (Pre-Resolvers) Format

You can reference S3 values as the source of your variables to use in your service with the `s3:bucketName/key` syntax.
It uses the deployment (provider) AWS credentials to access the S3 bucket.
For example:

```yml
service: new-service
provider: aws
functions:
  hello:
    name: ${s3:myBucket/myKey}-hello
    handler: handler.hello
```

In the above example, the value for `myKey` in the `myBucket` S3 bucket will be looked up and used to populate the variable.
Buckets from all regions can be used without any additional specification due to AWS S3 global strategy.
