<!--
title: Serverless Framework - AWS Lambda Events - SNS
description: Setting up AWS SNS Events with AWS Lambda via the Serverless Framework
short_title: AWS Lambda Events - SNS
keywords:
  [
    'Serverless',
    'Framework',
    'AWS',
    'Lambda',
    'Events',
    'AWS SNS Events',
    'Serverless AWS Lambda SNS',
    'AWS Lambda SNS Integration',
    'AWS Lambda Event Sources',
    'AWS SNS Lambda Trigger',
    'AWS SNS Topics',
  ]
-->

<!-- DOCS-SITE-LINK:START automatically generated  -->

### [Read this on the main serverless docs site](https://www.serverless.com/framework/docs/providers/aws/events/sns)

<!-- DOCS-SITE-LINK:END -->

# SNS

In the following example we create a new SNS topic with the name `dispatch` which is bound to the `dispatcher` function. The function will be called every time a message is sent to the `dispatch` topic.

```yml
functions:
  dispatcher:
    handler: dispatcher.dispatch
    events:
      - sns: dispatch
```

You're also able to add the same SNS topic to multiple functions:

```yml
functions:
  dispatcher:
    handler: dispatcher.dispatch
    events:
      - sns: dispatch
  dispatcher2:
    handler: dispatcher2.dispatch
    events:
      - sns: dispatch
```

This will run both functions for a message sent to the dispatch topic.

## Using a pre-existing topic

If an `arn:` is specified, the framework will give permission to the topic to invoke the function and subscribe the function to the topic.

```yml
functions:
  dispatcher:
    handler: dispatcher.dispatch
    events:
      - sns: arn:xxx
```

```yml
functions:
  dispatcher:
    handler: dispatcher.dispatch
    events:
      - sns:
          arn: arn:xxx
```

Or with intrinsic CloudFormation function like `Fn::Join`, `Fn::GetAtt`, or `Fn::Ref` (or their shorthand counterparts).
**Note:** The arn can be in a different region to enable cross region invocation

```yml
functions:
  dispatcher:
    handler: dispatcher.dispatch
    events:
      - sns:
          arn:
            Fn::Join:
              - ':'
              - - 'arn:aws:sns'
                - Ref: 'AWS::Region'
                - Ref: 'AWS::AccountId'
                - 'MyCustomTopic'
          topicName: MyCustomTopic
```

If your SNS topic doesn't yet exist but is defined in the serverless.yml file you're editing, you'll need to use `Fn::Ref` or `!Ref` to get the ARN. Do not build a string as in the above example!

```yml
functions:
  dispatcher:
    handler: dispatcher.dispatch
    events:
      - sns:
          arn: !Ref SuperTopic
          topicName: MyCustomTopic

resources:
  Resources:
    SuperTopic:
      Type: AWS::SNS::Topic
      Properties:
        TopicName: MyCustomTopic
```

**Note:** If an `arn` string is specified but not a `topicName`, the last substring starting with `:` will be extracted as the `topicName`. If an `arn` object is specified, `topicName` must be specified as a string, used only to name the underlying Cloudformation mapping resources. You can take advantage of this behavior when subscribing to multiple topics with the same name in different regions/accounts to avoid collisions between Cloudformation resource names.

```yml
functions:
  hello:
    handler: handler.run
    events:
      - sns:
          arn: arn:aws:sns:us-east-1:00000000000:topicname
          topicName: topicname-account-1-us-east-1
      - sns:
          arn: arn:aws:sns:us-east-1:11111111111:topicname
          topicName: topicname-account-2-us-east-1
```

## Setting a display name

This event definition ensures that the `aggregator` function gets called every time a message is sent to the
`aggregate` topic. `Data aggregation pipeline` will be shown in the AWS console so that the user can understand what the
SNS topic is used for.

```yml
functions:
  aggregator:
    handler: aggregator.handler
    events:
      - sns:
          topicName: aggregate
          displayName: Data aggregation pipeline
```

## Using a KMS key for created topics

When the framework creates a topic (via `topicName`), you can supply a KMS key to encrypt it:

- Set `kmsMasterKeyId` to a key alias or ARN (e.g., `alias/aws/sns` for the AWS-managed SNS key).
- This applies only when the topic is created by the framework. It does not modify pre-existing topics referenced by `arn`.

Examples:

Using the AWS-managed SNS key (`alias/aws/sns`):

```yml
functions:
  dispatcher:
    handler: dispatcher.handler
    events:
      - sns:
          topicName: dispatcher
          kmsMasterKeyId: alias/aws/sns
```

Using a custom KMS key alias:

```yml
functions:
  dispatcher:
    handler: dispatcher.handler
    events:
      - sns:
          topicName: dispatcher
          kmsMasterKeyId: alias/my-sns-key
```

Using a custom KMS key ARN:

```yml
functions:
  dispatcher:
    handler: dispatcher.handler
    events:
      - sns:
          topicName: dispatcher
          kmsMasterKeyId: arn:aws:kms:us-east-1:111111111111:key/12345678-9abc-def0-1234-56789abcdef1
```

Using a CloudFormation reference (for a KMS key defined in the same stack):

```yml
functions:
  dispatcher:
    handler: dispatcher.handler
    events:
      - sns:
          topicName: dispatcher
          kmsMasterKeyId: !Ref MyKmsKey

resources:
  Resources:
    MyKmsKey:
      Type: AWS::KMS::Key
      Properties:
        KeyPolicy: { /* ... */ }
```

## Setting a filter policy

This event definition creates an SNS topic which subscription uses a filter policy. The filter policy filters out messages that don't have attribute key `pet` with value `dog` or `cat`.

```yml
functions:
  pets:
    handler: pets.handler
    events:
      - sns:
          topicName: pets
          filterPolicy:
            pet:
              - dog
              - cat
```

## Setting a filter policy scope

This event definition specifies the scope for a filter policy by setting it to one of the following values:

- `MessageAttributes`: The filter policy will be applied to the message attributes. This is the default if no scope is defined for an existing filter policy.
- `MessageBody`: The filter policy will be applied to the message body.

Note: If no filter policy scope is defined for an existing filter policy, the scope defaults to MessageAttributes. For more information, see [SNS Message Filtering](https://docs.aws.amazon.com/sns/latest/dg/sns-message-filtering.html).

```yml
functions:
  pets:
    handler: pets.handler
    events:
      - sns:
          topicName: pets
          filterPolicyScope: MessageBody
          filterPolicy:
            pet:
              - dog
              - cat
```

## Setting a redrive policy

This event definition creates an SNS topic that sends messages to a Dead Letter Queue (defined by its ARN) when the associated lambda is not available. In this example, messages that aren't delivered to the `dispatcher` Lambda (because the lambda service is down or irresponsive) will end in `myDLQ`.

```yml
functions:
  dispatcher:
    handler: dispatcher.handler
    events:
      - sns:
          topicName: dispatcher
          redrivePolicy:
            deadLetterTargetArn: arn:aws:sqs:us-east-1:11111111111:myDLQ
```

To define the Dead Letter Queue, you can alternatively use the the resource name with `deadLetterTargetRef`

```yml
functions:
  dispatcher:
    handler: dispatcher.handler
    events:
      - sns:
          topicName: dispatcher
          redrivePolicy:
            deadLetterTargetRef: myDLQ

resources:
  Resources:
    myDLQ:
      Type: AWS::SQS::Queue
      Properties:
        QueueName: myDLQ
```

Or if you want to use values from other stacks, you can
also use `deadLetterTargetImport` to define the DLQ url and arn with exported values

```yml
functions:
  dispatcher:
    handler: dispatcher.handler
    events:
      - sns:
          topicName: dispatcher
          redrivePolicy:
            deadLetterTargetImport:
              arn: MyShared-DLQArn
              url: MyShared-DLQUrl
```
